<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class Order extends Model
{
    use HasFactory;

    protected $table = 'orders';
    protected $guarded = ['id'];

    public function scopeOwn($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function shipping()
    {
        return $this->belongsTo(StoreShipping::class, 'shipping_id');
    }

    public function store()
    {
        return $this->belongsTo(Store::class, 'store_id');
    }
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function gateway()
    {
        return $this->belongsTo(UserGateway::class, 'gateway_id');
    }

    public function userGateway()
    {
        return $this->belongsTo(UserGateway::class, 'gateway_id', 'id');
    }

    public function details()
    {
        return $this->hasMany(OrderDetail::class, 'order_id');
    }
    public function deposit(){
        return $this->belongsTo(Deposit::class, 'deposit_id');
    }
    public function transactional()
    {
        return $this->morphOne(Transaction::class, 'transactional');
    }

    public function stageClass($status)
    {
        return [
            '0' => 'warning',
            '1' => 'success',
            '2' => 'primary',
            '3' => 'info',
            '4' => 'danger',
        ][$status ?? 0];
    }

    public static function getStage($status)
    {
        return [
            '0' => 'Pending',
            '1' => 'Delivered',
            '2' => 'In Process',
            '3' => 'Out for Delivery',
            '4' => 'Canceled',
        ][$status ?? 0];
    }


    public static function boot(): void
    {
        parent::boot();
        static::saved(function () {
            Cache::forget('orderRecord');
        });

        static::creating(function (Order $order) {
            if (empty($order->utr)) {
                $order->utr = self::generateOrderNumber();
            }
        });
    }

    public static function generateOrderNumber()
    {
        return DB::transaction(function () {
            $lastOrder = self::lockForUpdate()->orderBy('id', 'desc')->first();
            if ($lastOrder && isset($lastOrder->utr)) {
                $lastOrderNumber = (int)filter_var($lastOrder->utr, FILTER_SANITIZE_NUMBER_INT);
                $newOrderNumber = $lastOrderNumber + 1;
            } else {
                $newOrderNumber = strRandomNum(12);
            }

            // Check again to ensure the new trx_id doesn't already exist (extra safety)
            while (self::where('utr', 'O'.$newOrderNumber)->exists()) {
                $newOrderNumber = (int)$newOrderNumber + 1;
            }
            return 'O' . $newOrderNumber;
        });
    }
}
