<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StoreProduct;
use App\Models\StoreProductOptional;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Yajra\DataTables\Facades\DataTables;

class MenuController extends Controller
{
    public function menu($store_id = null)
    {
        $storeProductRecord = \Cache::get('storeProductRecord');
        if (!$storeProductRecord) {
            $storeProductRecord = StoreProduct::selectRaw('COUNT(id) AS totalStoreProduct')
                ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS activeStoreProduct')
                ->selectRaw('(COUNT(CASE WHEN status = 1 THEN id END) / COUNT(id)) * 100 AS activeStoreProductPercent')
                ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS inactiveStoreProduct')
                ->selectRaw('(COUNT(CASE WHEN status = 0 THEN id END) / COUNT(id)) * 100 AS inactiveStoreProductPercent')
                ->selectRaw('(SELECT COUNT(id) FROM store_products WHERE DATE(created_at) = CURDATE()) AS todayStoreProduct')
                ->selectRaw('(SELECT COUNT(id) FROM store_products WHERE MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE())) AS thisMonthStoreProduct')
                ->selectRaw('((SELECT COUNT(id) FROM store_products WHERE DATE(created_at) = CURDATE()) / NULLIF(COUNT(id), 0)) * 100 AS todayStoreProductPercent')
                ->selectRaw('((SELECT COUNT(id) FROM store_products WHERE MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE())) / NULLIF(COUNT(id), 0)) * 100 AS thisMonthStoreProductPercent')
                ->get()
                ->toArray();
            \Cache::put('storeProductRecord', $storeProductRecord);
        }

        $data['storeProductRecord'] = $storeProductRecord;
        $data['store_id'] = $store_id;
        return view('admin.menu.menu_item', $data);
    }

    public function menuSearch(Request $request, $store_id = null)
    {
        $search = $request->search['value'] ?? null;
        $filterInput = $request->filterInput;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $items = StoreProduct::query()
            ->with(['user:id,firstname,lastname,username,email,image,image_driver','category:id,name','productStores'])
            ->orderBy('id', 'DESC')
            ->when( $store_id != 'null', function ($query) use ($store_id) {
                $query->whereHas('productStores', function ($qq) use ($store_id) {
                    $qq->where('store_id', $store_id);
                });
            })
            ->when(!empty($search), function ($query) use ($search) {
                return $query->where('name', 'LIKE', "%$search%")
                    ->orWhere('sku', 'LIKE', "%$search%")
                    ->orWhere('tag', 'LIKE', "%$search%")
                    ->orWhereHas('user', function ($q2) use ($search) {
                        $q2->where('firstname', 'LIKE', "%$search%")
                            ->orWhere('lastname', 'LIKE', "%$search%")
                            ->orWhere('username', 'LIKE', "%$search%")
                            ->orWhere('email', 'LIKE', "%$search%");
                    })
                    ->orWhereHas('category', function ($q3) use ($search) {
                        $q3->where('name', 'LIKE', "%$search%");
                    });
            })
            ->when(!empty($filterInput), function ($query) use ($filterInput) {
                return $query->where('name', 'LIKE', "%$filterInput%")
                    ->orWhere('sku', 'LIKE', "%$filterInput%")
                    ->orWhere('tag', 'LIKE', "%$filterInput%")
                    ->orWhereHas('user', function ($q2) use ($filterInput) {
                        $q2->where('firstname', 'LIKE', "%$filterInput%")
                            ->orWhere('lastname', 'LIKE', "%$filterInput%")
                            ->orWhere('username', 'LIKE', "%$filterInput%")
                            ->orWhere('email', 'LIKE', "%$filterInput%");
                    })
                    ->orWhereHas('category', function ($q3) use ($filterInput) {
                        $q3->where('name', 'LIKE', "%$filterInput%");
                    });
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            });

        return DataTables::of($items)
            ->addColumn('no', function ($item) {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('menu', function ($item) {
                return '<a class="d-flex align-items-center me-2" href="javascript:void(0)">
                            <div class="flex-shrink-0">
                                <div class="avatar avatar-sm avatar-circle">
                                    <img class="avatar-img" src="' . getFile($item->driver,$item->image) . '" alt="Image">
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                              <h5 class="text-hover-primary mb-0">' . $item->name . '</h5>
                              <span class="fs-6 text-body">' . $item->sku . '</span>
                            </div>
                      </a>';
            })

            ->addColumn('category', function ($item) {
                return optional($item->category)->name;
            })
            ->addColumn('price', function ($item) {
                return $item->variants ? '-' : currencyPosition(getAmount($item->price));
            })
            ->addColumn('owner', function ($item) {
                $url = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="'.$url.'">
                            <div class="flex-shrink-0">
                              ' . $item->user->profilePicture() . '
                            </div>
                            <div class="flex-grow-1 ms-3">
                              <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                              <span class="fs-6 text-body">' . optional($item->user)->email . '</span>
                            </div>
                      </a>';
            })
            ->addColumn('status', function ($item) {
                $badgeClass = $item->status == 1 ? 'success text-success' : 'danger text-danger';
                $legendBgClass = $item->status == 1 ? 'success' : 'danger';
                $status = $item->status == 1 ? 'Active' : 'Inactive';
                return '<span class="badge bg-soft-'.$badgeClass.'"><span class="legend-indicator bg-'.$legendBgClass.'"></span>'.$status.'</span>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at);
            })
            ->addColumn('action', function ($item) {
                return dateTime($item->created_at);
            })
            ->addColumn('action', function ($item) {
                $url = route('admin.menu.details', $item->id);
                return "<a href='".$url."' class='btn btn-white btn-sm edit_btn'><i class='bi-eye fill me-1'></i> Details</a>";
            })
            ->rawColumns(['menu','owner','category','price','status','created_at','action'])
            ->make(true);
    }

    public function menuDetails($id)
    {
        $data['product'] = StoreProduct::with(['productStores.store'])->findOrFail($id);
        return view('admin.menu.menu_details', $data);
    }

    public function optional()
    {
        $storeProductOptionalRecord = \Cache::get('storeProductOptionalRecord');
        if (!$storeProductOptionalRecord) {
            $storeProductOptionalRecord = StoreProductOptional::selectRaw('COUNT(id) AS totalStoreOptionalProduct')
                ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS activeStoreOptionalProduct')
                ->selectRaw('(COUNT(CASE WHEN status = 1 THEN id END) / COUNT(id)) * 100 AS activeStoreOptionalProductPercent')
                ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS inactiveStoreOptionalProduct')
                ->selectRaw('(COUNT(CASE WHEN status = 0 THEN id END) / COUNT(id)) * 100 AS inactiveStoreOptionalProductPercent')
                ->selectRaw('(SELECT COUNT(id) FROM store_product_optionals WHERE DATE(created_at) = CURDATE()) AS todayStoreOptionalProduct')
                ->selectRaw('(SELECT COUNT(id) FROM store_product_optionals WHERE MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE())) AS thisMonthStoreOptionalProduct')
                ->selectRaw('((SELECT COUNT(id) FROM store_product_optionals WHERE DATE(created_at) = CURDATE()) / NULLIF(COUNT(id), 0)) * 100 AS todayStoreOptionalProductPercent')
                ->selectRaw('((SELECT COUNT(id) FROM store_product_optionals WHERE MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE())) / NULLIF(COUNT(id), 0)) * 100 AS thisMonthStoreOptionalProductPercent')
                ->get()
                ->toArray();
            \Cache::put('storeProductOptionalRecord', $storeProductOptionalRecord);
        }

        $data['storeProductOptionalRecord'] = $storeProductOptionalRecord;
        return view('admin.menu.optional', $data);
    }

    public function optionalSearch(Request $request)
    {
        $search = $request->search['value'] ?? null;
        $filterInput = $request->filterInput;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $purchasePlan = StoreProductOptional::query()
            ->with(['user:id,firstname,lastname,username,email,image,image_driver'])
            ->orderBy('id', 'DESC')
            ->when(!empty($search), function ($query) use ($search) {
                return $query->where('name', 'LIKE', "%$search%")
                    ->orWhereHas('user', function ($q2) use ($search) {
                        $q2->where('firstname', 'LIKE', "%$search%")
                            ->orWhere('lastname', 'LIKE', "%$search%")
                            ->orWhere('username', 'LIKE', "%$search%")
                            ->orWhere('email', 'LIKE', "%$search%");
                    });
            })
            ->when(!empty($filterInput), function ($query) use ($filterInput) {
                return $query->where('name', 'LIKE', "%$filterInput%")
                    ->orWhereHas('user', function ($q2) use ($filterInput) {
                        $q2->where('firstname', 'LIKE', "%$filterInput%")
                            ->orWhere('lastname', 'LIKE', "%$filterInput%")
                            ->orWhere('username', 'LIKE', "%$filterInput%")
                            ->orWhere('email', 'LIKE', "%$filterInput%");
                    });
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            });

        return DataTables::of($purchasePlan)
            ->addColumn('no', function ($item) {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('menu', function ($item) {
                return '<a class="d-flex align-items-center me-2" href="javascript:void(0)">
                            <div class="flex-shrink-0">
                                <div class="avatar avatar-sm avatar-circle">
                                    <img class="avatar-img" src="' . getFile($item->driver,$item->image) . '" alt="Image">
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                              <h5 class="text-hover-primary mb-0">' . $item->name . '</h5>
                              <span class="fs-6 text-body">' . currencyPosition(getAmount($item->price)) . '</span>
                            </div>
                      </a>';
            })

            ->addColumn('owner', function ($item) {
                $url = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="'.$url.'">
                            <div class="flex-shrink-0">
                              ' . $item->user->profilePicture() . '
                            </div>
                            <div class="flex-grow-1 ms-3">
                              <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                              <span class="fs-6 text-body">' . optional($item->user)->email . '</span>
                            </div>
                      </a>';
            })
            ->addColumn('status', function ($item) {
                $badgeClass = $item->status == 1 ? 'success text-success' : 'danger text-danger';
                $legendBgClass = $item->status == 1 ? 'success' : 'danger';
                $status = $item->status == 1 ? 'Active' : 'Inactive';
                return '<span class="badge bg-soft-'.$badgeClass.'"><span class="legend-indicator bg-'.$legendBgClass.'"></span>'.$status.'</span>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at);
            })
            ->rawColumns(['menu','owner','status','created_at'])
            ->make(true);
    }
}
