<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\StoreProductOptional;
use App\Traits\Upload;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StoreProductOptionalController extends Controller
{
    use Upload;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index(Request $request)
    {
        $search = $request->all();
        $dateSearch = Carbon::parse($request->created_at);

        $data['optionalProducts'] = StoreProductOptional::own($this->user->id)
            ->when(isset($search['name']), function ($query) use ($search) {
                $query->where('name', 'LIKE', '%' . $search['name'] . '%');
            })
            ->when(isset($search['min_price']), function ($query) use ($search) {
                $query->where('price', '>', $search['min_price']);
            })
            ->when(isset($search['max_price']), function ($query) use ($search) {
                $query->where('price', '<', $search['max_price']);
            })
            ->when(isset($search['created_at']), function ($query) use ($dateSearch) {
                return $query->whereDate("created_at", $dateSearch);
            })
            ->when(isset($search['status']), function ($query) use ($search) {
                if ($search['status'] == 'active') {
                    $query->where('status', 1);
                } else {
                    $query->where('status', 0);
                }
            })
            ->toBase()
            ->orderBy('id', 'desc')->paginate(basicControl()->paginate);
        return view($this->theme . 'user.store.product.optional.index', $data);
    }

    public function create(Request $request)
    {
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.store.product.optional.create');
        }
        if ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|min:2',
                'price' => 'required|numeric|min:0|not_in:0',
                'image' => 'required'
            ]);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            $optionalProduct = new StoreProductOptional();
            $optionalProduct->user_id = $this->user->id;
            $optionalProduct->name = $request['name'];
            $optionalProduct->price = $request['price'];
            $optionalProduct->status = $request['status'];
            if ($request->hasFile('image')) {
                try {
                    $image = $this->fileUpload($request->image, config('filelocation.optionalProduct.path'), null, config('filelocation.optionalProduct.size'), 'webp', 99);
                    if ($image) {
                        $optionalProduct->image = $image['path'];
                        $optionalProduct->driver = $image['driver'];
                    }
                } catch (\Exception $e) {
                    return back()->withInput()->with('error', 'Image could not be uploaded');
                }
            }
            $optionalProduct->save();
            return back()->with('success', 'Add ons Has Been Created');
        }
    }

    public function edit(Request $request, $id)
    {
        $data['optionalProduct'] = StoreProductOptional::own($this->user->id)->findOrFail($id);
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.store.product.optional.edit', $data);
        }
        if ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|min:2',
                'price' => 'required|numeric|min:0|not_in:0',
            ]);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            $data['optionalProduct']->name = $request['name'];
            $data['optionalProduct']->price = $request['price'];
            $data['optionalProduct']->status = $request['status'];
            if ($request->hasFile('image')) {
                try {
                    $image = $this->fileUpload($request->image, config('filelocation.optionalProduct.path'), null, config('filelocation.optionalProduct.size'), 'webp', 99,$data['optionalProduct']->image,$data['optionalProduct']->driver);
                    if ($image) {
                        $data['optionalProduct']->image = $image['path'];
                        $data['optionalProduct']->driver = $image['driver'];
                    }
                } catch (\Exception $e) {
                    return back()->withInput()->with('alert', 'Image could not be uploaded');
                }
            }
            $data['optionalProduct']->save();
            return back()->with('success', 'Updated Successfully');
        }
    }

    public function delete($id)
    {
        $optionalProduct = StoreProductOptional::own($this->user->id)->findOrFail($id);
        if ($optionalProduct->extra_products()) {
            return back()->with('error', 'Add ons has lot of menu');
        }
        $this->fileDelete($optionalProduct->driver,$optionalProduct->image);
        $optionalProduct->delete();
        return back()->with('success', 'Add ons Has Been Deleted');
    }
}
