<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Language;
use App\Models\Schedule;
use App\Models\ScheduleDetails;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class ScheduleController extends Controller
{
    public function index()
    {
        return view('admin.schedule.index');
    }

    public function search(Request $request)
    {
        $search = $request->search['value'] ?? null;
        $filterName = $request->filterInput;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $plans = Schedule::query()->with('details')->orderBy('id', 'DESC')
            ->when(!empty($search), function ($query) use ($search) {
                return $query->where('days', 'LIKE', "%$search%")
                        ->orWhereHas('details', function ($q) use ($search) {
                            $q->where('name', 'LIKE', "%$search%");
                        });
            })
            ->when(isset($filterName) && !empty($filterName), function ($query) use ($filterName) {
                return $query->where('days', 'LIKE', "%$filterName%")
                    ->orWhereHas('details', function ($q) use ($filterName) {
                        $q->where('name', 'LIKE', "%$filterName%");
                    });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus == 'all') {
                    return $query->where('status', '!=', null);
                }
                return $query->where('status', $filterStatus);
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            });

        return DataTables::of($plans)
            ->addColumn('checkbox', function ($item) {
                return ' <input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('name', function ($item) {
                return optional($item->details)->name;
            })
            ->addColumn('duration', function ($item) {
                $duration = $item->days > 1 ? $item->days . ' Days' : $item->days.' Day';
                return $duration;
            })
            ->addColumn('status', function ($item) {
                if ($item->status == 1) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('Active') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-danger text-danger">
                    <span class="legend-indicator bg-danger"></span>' . trans('Inactive') . '
                  </span>';
                }
            })
            ->addColumn('action', function ($item) {
                $editUrl = route('admin.schedule.edit', $item->id);
                $deleteUrl = route('admin.schedule.delete', $item->id);
                return '<div class="btn-group" role="group">
                        <a href="' . $editUrl . '" class="btn btn-white btn-sm edit_user_btn">
                        <i class="bi-pencil-fill me-1"></i> ' . trans("Edit") . '
                      </a>
                    <div class="btn-group">
                      <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                      <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                      <a href="javascript:void(0)" class="dropdown-item text-danger delete-button" data-bs-target="#deleteModal" data-bs-toggle="modal"
                            data-name="' . $item->details->name . '"` data-route="' . $deleteUrl . '">
                        <i class="bi-trash-fill me-1"></i> ' . trans("Delete") . '
                      </a>
                      </div>
                    </div>
                  </div>';
            })->rawColumns(['checkbox', 'name', 'duration', 'status', 'action'])
            ->make(true);
    }

    public function create()
    {
        $languages = Language::all();
        return view('admin.schedule.create', compact('languages'));
    }

    public function store(Request $request, $language)
    {
        $rules = [
            'name.*' => 'required|max:191|unique:schedule_details,name,NULL,id,language_id,' . $language,
            'days' => 'required'
        ];
        $messages = [
            'name.*.required' => 'Name is required.',
            'name.*.max' => 'The name cannot exceed :max characters.',
            'name.*.unique' => 'The name has already been taken.',
            'days.required' => 'Days is required.',
        ];
        $validate = Validator::make($request->all(), $rules, $messages);

        if ($validate->fails()) {
            return back()->withInput()->withErrors($validate);
        }

        DB::beginTransaction();
        try {
            $schedule = new Schedule();
            $schedule->days = $request->days ?? 0;
            $schedule->status = $request->status ?? 1;
            $schedule->save();
            $schedule->details()->create([
                'language_id' => $language,
                'name' => $request->name[$language]
            ]);
            DB::commit();
            return redirect()->back()->with('success', 'Schedule has been created Successfully.');
        }catch (\Exception $exception){
            DB::rollBack();
            return back()->with('error', $exception->getMessage());
        }
    }

    public function edit($id)
    {
        $data['id'] = $id;
        $data['languages'] = Language::all();
        $data['scheduleDetails'] = ScheduleDetails::with('schedule')->where('schedule_id', $id)->get()->groupBy('language_id');
        return view('admin.schedule.edit', $data);
    }

    public function update(Request $request, $id, $language)
    {
        $rules = [
            'name.*' => 'required|max:191|unique:schedule_details,name,' . $id . ',schedule_id,language_id,' . $language,
            'days' => 'sometimes|required',
        ];
        $messages = [
            'name.*.required' => 'Name is required.',
            'name.*.max' => 'The name cannot exceed :max characters.',
            'name.*.unique' => 'The name has already been taken.'
        ];

        $validate = Validator::make($request->all(), $rules, $messages);

        if ($validate->fails()) {
            return back()->withInput()->withErrors($validate);
        }

        DB::beginTransaction();
        try {
            $schedule = Schedule::findOrFail($id);
            $defaultLanguage = Language::select('id','default_status')->findOrFail($language);

            if ($defaultLanguage->default_status){
                $schedule->days = $request->days ?? 0;
                $schedule->status = $request->status ?? 1;
                $schedule->save();
            }

            $schedule->details()->updateOrCreate([
                'language_id' => $language
            ],
                [
                    'name' => $request->name[$language]
                ]
            );

            DB::commit();
            return redirect()->back()->with('success', 'Schedule has been Updated Successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return back()->with('error', $exception->getMessage());
        }
    }

    public function delete($id)
    {
        $schedule = Schedule::findOrFail($id);
        $schedule->details()->delete();
        $schedule->delete();
        return back()->with('success', 'Schedule has been deleted successfully.');
    }

    public function deleteMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select Data.');
            return response()->json(['error' => 1]);
        } else {
            Schedule::whereIn('id', $request->strIds)->get()->map(function ($schedule) {
                $schedule->details()->delete();
                $schedule->delete();
            });
            session()->flash('success', 'Schedule has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
