<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Language;
use App\Models\Plan;
use App\Models\PlanDetails;
use App\Models\PlanSchedule;
use App\Models\Schedule;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class PlanController extends Controller
{
    public function index()
    {
        return view('admin.plan.index');
    }

    public function search(Request $request)
    {
        $search = $request->search['value'] ?? null;
        $filterName = $request->filterInput;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $plans = Plan::query()->with('details')->orderBy('id', 'DESC')
            ->when(!empty($search), function ($query) use ($search) {
                return $query->whereHas('details', function ($q) use ($search) {
                        $q->where('plan_name', 'LIKE', "%$search%")
                        ->orWhere('badge_name', 'LIKE', "%$search%");
                    });
            })
            ->when(isset($filterName) && !empty($filterName), function ($query) use ($filterName) {
                return $query->whereHas('details', function ($q) use ($filterName) {
                        $q->where('plan_name', 'LIKE', "%$filterName%")
                        ->orWhere('badge_name', 'LIKE', "%$filterName%");
                    });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus == 'all') {
                    return $query->where('status', '!=', null);
                }
                return $query->where('status', $filterStatus);
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            });

        return DataTables::of($plans)
            ->addColumn('checkbox', function ($item) {
                return ' <input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('name', function ($item) {
                return optional($item->details)->plan_name;
            })
            ->addColumn('stores', function ($item) {
                return '<span class="badge bg-soft-secondary text-secondary">' . $item->num_stores . '</span>';
            })
            ->addColumn('products', function ($item) {
                return '<span class="badge bg-soft-secondary text-secondary">' . $item->product_qty . '</span>';
            })
            ->addColumn('orders', function ($item) {
                return '<span class="badge bg-soft-secondary text-secondary">' . $item->order_num . '</span>';
            })
            ->addColumn('telegram_bot', function ($item) {
                $text = $item->telegram_bot == 1 ? 'Yes' : 'No';
                $class = $item->telegram_bot == 1 ? 'success' : 'danger';
                return '<span class="badge bg-soft-'.$class.' text-'.$class.'">
                        <span class="legend-indicator bg-'.$class.'"></span>' . trans($text) . '
                      </span>';
            })
            ->addColumn('messenger_bot', function ($item) {
                $text = $item->messenger_bot == 1 ? 'Yes' : 'No';
                $class = $item->messenger_bot == 1 ? 'success' : 'danger';
                return '<span class="badge bg-soft-'.$class.' text-'.$class.'">
                        <span class="legend-indicator bg-'.$class.'"></span>' . trans($text) . '
                      </span>';
            })
            ->addColumn('whatsapp_bot', function ($item) {
                $text = $item->whatsapp_bot == 1 ? 'Yes' : 'No';
                $class = $item->whatsapp_bot == 1 ? 'success' : 'danger';
                return '<span class="badge bg-soft-'.$class.' text-'.$class.'">
                        <span class="legend-indicator bg-'.$class.'"></span>' . trans($text) . '
                      </span>';
            })
            ->addColumn('status', function ($item) {
                $text = $item->status == 1 ? 'Active' : 'Inactive';
                $class = $item->status == 1 ? 'success' : 'danger';
                return '<span class="badge bg-soft-'.$class.' text-'.$class.'">
                        <span class="legend-indicator bg-'.$class.'"></span>' . trans($text) . '
                      </span>';
            })
            ->addColumn('action', function ($item) {
                $editUrl = route('admin.plan.edit', $item->id);
                $deleteUrl = route('admin.plan.delete', $item->id);
                return '<div class="btn-group" role="group">
                        <a href="' . $editUrl . '" class="btn btn-white btn-sm edit_user_btn">
                        <i class="bi-pencil-fill me-1"></i> ' . trans("Edit") . '
                      </a>
                    <div class="btn-group">
                      <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                      <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                      <a href="javascript:void(0)" class="dropdown-item text-danger delete-button" data-bs-target="#deleteModal" data-bs-toggle="modal"
                            data-name="' . optional($item->details)->plan_name . '"` data-route="' . $deleteUrl . '">
                        <i class="bi-trash-fill me-1"></i> ' . trans("Delete") . '
                      </a>
                      </div>
                    </div>
                  </div>';
            })->rawColumns(['checkbox', 'name', 'stores', 'products','orders','telegram_bot','messenger_bot','whatsapp_bot','status', 'action'])
            ->make(true);
    }

    public function create()
    {
        $data['languages'] = Language::all()->toBase();
        $data['schedules'] = Schedule::with('details')->where('status',1)->get();
        return view('admin.plan.create', $data);
    }

    public function store(Request $request, $language)
    {
        $rules = [
            'plan_name.*' => 'required|max:191|unique:plan_details,plan_name,NULL,id,language_id,' . $language,
            'badge_name.*' => 'nullable|string|max:191',
            'num_stores' => 'sometimes|required|min:1',
            'product_qty' => 'sometimes|required|min:1',
            'order_num' => 'sometimes|required|min:1',
            'status' => 'sometimes|required',
            'price' => 'sometimes|required|min:0',
            'schedule.*' => 'sometimes|required|integer|distinct',
            'note.*' => 'nullable|string|max:100',
        ];
        $messages = [
            'plan_name.*.required' => 'Plan Name is required.',
            'plan_name.*.max' => 'Plan Name cannot exceed :max characters.',
            'plan_name.*.unique' => 'Plan Name has already been taken.',
            'price.required' => 'Price is required.',
            'num_stores.required' => 'Number Of Stores is required.',
            'product_qty.required' => 'Quantity Of Menu is required.',
            'order_num.required' => 'Number Of Orders is required.',
        ];
        $validate = Validator::make($request->all(), $rules, $messages);

        if ($validate->fails()) {
            return back()->withInput()->withErrors($validate);
        }

        DB::beginTransaction();
        try {
            $plan = new Plan();
            $plan->num_stores = $request->num_stores;
            $plan->product_qty = $request->product_qty;
            $plan->order_num = $request->order_num;
            $plan->status = $request->status ?? 1;
            $plan->telegram_bot = $request->telegram_bot;
            $plan->messenger_bot = $request->messenger_bot;
            $plan->whatsapp_bot = $request->whatsapp_bot;
            $plan->save();
            $plan->details()->create([
                'language_id' => $language,
                'plan_name' => $request->plan_name[$language],
                'badge_name' => $request->badge_name[$language],
                'note' => $request->note[$language],
            ]);

            foreach ($request->schedule as $key => $value) {
                PlanSchedule::create([
                    'plan_id' => $plan->id,
                    'schedule_id' => (int) $value,
                    'price' => $request->price[$key],
                ]);
            }
            DB::commit();
            return redirect()->back()->with('success', 'Plan has been created Successfully.');
        }catch (\Exception $exception){
            DB::rollBack();
            return back()->with('error', $exception->getMessage());
        }
    }

    public function edit($id)
    {
        $data['id'] = $id;
        $data['languages'] = Language::all()->toBase();
        $data['schedules'] = Schedule::with('details')->where('status',1)->get();
        $data['planDetails'] = PlanDetails::with('plan.planSchedules')->where('plan_id', $id)->get()->groupBy('language_id');
        return view('admin.plan.edit', $data);
    }

    public function update(Request $request, $id, $language)
    {
        $rules = [
            'plan_name.*' => 'required|max:191|unique:plan_details,plan_name,'. $id .',plan_id,language_id,' . $language,
            'badge_name.*' => 'nullable|string|max:191',
            'num_stores' => 'sometimes|required|min:1',
            'product_qty' => 'sometimes|required|min:1',
            'order_num' => 'sometimes|required|min:1',
            'status' => 'sometimes|required',
            'price' => 'sometimes|required|min:0',
            'schedule.*' => 'sometimes|required|integer|distinct',
            'note.*' => 'nullable|string|max:100',
        ];
        $messages = [
            'plan_name.*.required' => 'Plan Name is required.',
            'plan_name.*.max' => 'Plan Name cannot exceed :max characters.',
            'plan_name.*.unique' => 'Plan Name has already been taken.',
            'price.required' => 'Price is required.',
            'num_stores.required' => 'Number Of Stores is required.',
            'product_qty.required' => 'Quantity Of Menu is required.',
            'order_num.required' => 'Number Of Orders is required.',
        ];

        $validate = Validator::make($request->all(), $rules, $messages);

        if ($validate->fails()) {
            return back()->withInput()->withErrors($validate);
        }

        DB::beginTransaction();
        try {
            $plan = Plan::findOrFail($id);
            $defaultLanguage = Language::select('id','default_status')->findOrFail($language);

            if ($defaultLanguage->default_status){
                $plan->num_stores = $request->num_stores;
                $plan->product_qty = $request->product_qty;
                $plan->order_num = $request->order_num;
                $plan->status = $request->status ?? 1;
                $plan->telegram_bot = $request->telegram_bot;
                $plan->messenger_bot = $request->messenger_bot;
                $plan->whatsapp_bot = $request->whatsapp_bot;
                $plan->save();
                $plan->planSchedules()->delete();
                foreach ($request->schedule as $key => $value) {
                    PlanSchedule::create([
                        'plan_id' => $plan->id,
                        'schedule_id' => (int) $value,
                        'price' => $request->price[$key],
                    ]);
                }
            }

            $plan->details()->updateOrCreate([
                'language_id' => $language
            ],
                [
                    'plan_name' => $request->plan_name[$language],
                    'badge_name' => $request->badge_name[$language],
                    'note' => $request->note[$language],
                ]
            );

            DB::commit();
            return redirect()->back()->with('success', 'Plan has been Updated Successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return back()->with('error', $exception->getMessage());
        }
    }

    public function delete($id)
    {
        $plan = Plan::findOrFail($id);
        DB::beginTransaction();
        try {
            $plan->details()->delete();
            $plan->planSchedules()->delete();
            $plan->delete();
            DB::commit();
        }catch (\Exception $exception){
            DB::rollBack();
            return back()->with('error', $exception->getMessage());
        }
        return back()->with('success', 'Schedule has been deleted successfully.');
    }

    public function deleteMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select Data.');
            return response()->json(['error' => 1]);
        } else {
            Plan::whereIn('id', $request->strIds)->get()->map(function ($item) {
                $item->details()->delete();
                $item->planSchedules()->delete();
                $item->delete();
            });
            session()->flash('success', 'Paln has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
